C      *****************************************************************
C      * Subroutine DeCntrl                                            *
C      * Controls decomposition and updates associated variables.      *
C      * This is done for all layers in the soil.                      *
C      *                                                               *
C      * Variables:                                                    *
C      * BrdthMoist-Input - The soil moisture breadth parameter.  Units*
C      *                    are mm available water.                    *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * CAmt      -Output- A matrix, of size 4 x NLayers, of the      *
C      *                    amount of carbon in cellulose, lignin,     *
C      *                    extractives and humus for each layer.      *
C      *                    Units are g C/m**2.                        *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * Carb      -Input - The array index for carbon.  Unitless.     *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * CatchArea -Input - A vector, of size NLayers, of the catchment*
C      *                    area for each layer.  Input from TREGRO, or*
C      *                    read into the model by the routine InParam.*
C      *                    Units are m**2.                            *
C      *                    (Common block VModel, file VModel.f)       *
C      * CE        -Input - The array index for cellulose to           *
C      *                    extractives.  Unitless.                    *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * Cell      -Input - The array index for cellulose.  Unitless.  *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * CellN     -Local - The amount of nitrogen in cellulose in the *
C      *                    leaves. Units are g N.                     *
C      * CtoN      -Input - A vector, of size four, of the carbon to   *
C      *                    nitrogen ratio for cellulose, extractives, *
C      *                    lignin, and humus.  Unitless.              *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * EL        -Input - The array index for extractives to lignin. *
C      *                    Unitless.                                  *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * Extr      -Input - The array index for extractives.  Unitless.*
C      *                    (Common block VDecomp, file VModel.f)      *
C      * Fracts    -Input - A three dimesional matrix, of size 3 by 2  *
C      *                    by NLayers, which holds the fraction of    *
C      *                    carbon and nitrogen in the different root  *
C      *                    parts for each layer, as follows:          *
C      *                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
C      *                    ~                 CARBON    NITROGEN    ~  *
C      *                    ~               (fraction) (fraction)   ~  *
C      *                    ~    CELLULOSE      ##        ##        ~  *
C      *                    ~    EXTRACTIVES    ##        ##        ~  *
C      *                    ~    LIGNIN         ##        ##        ~  *
C      *                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
C      *                    Update 3/93 => no more fraction for        *
C      *                    nitrogen.  Doesn't take up much room so am *
C      *                    leaving it in for now.  Unitless.          *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * gNperMole -Local - The grams of nitrogen per mole. Units are  *
C      *                    g N/mole.                                  *
C      * HE        -Input - The array index for humus to extractives.  *
C      *                    Unitless.                                  *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * Humu      -Input - The array index for humus.  Unitless.      *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * ILC       -Input - A vector, of size 2, of ILC shielding      *
C      *                    parameters for cellulose and extractives.  *
C      *                    Unitless.                                  *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * KT        -In/Out- A matrix, of size NComp by NLayer, of the  *
C      *                    known total amount for each component, for *
C      *                    each layer.  The total includes the amount *
C      *                    of the component as a free component and as*
C      *                    part of all dependent species.  Units are  *
C      *                    mol.                                       *
C      *                    (Common block VModel, file VModel.f)       *
C      * LE        -Input - The array index for lignin to extractives. *
C      *                    Unitless.                                  *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * LfFracts  -      - A three dimesional matrix, of size 3 by 2  *
C      *                    by NLayers, which holds the fraction of    *
C      *                    carbon and nitrogen in the senesced leaves,*
C      *                    as follows:                                *
C      *                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
C      *                    ~                 CARBON    NITROGEN    ~  *
C      *                    ~               (fraction) (fraction)   ~  *
C      *                    ~    CELLULOSE      ##        ##        ~  *
C      *                    ~    EXTRACTIVES    ##        ##        ~  *
C      *                    ~    LIGNIN         ##        ##        ~  *
C      *                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
C      *                    Update 3/93 => no more fraction for        *
C      *                    nitrogen.  Doesn't take up much room so am *
C      *                    leaving it in for now.  Unitless.          *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * LfNut     -Input - A vector, of size NComp, of the amount of  *
C      *                    each nutrient in the leaves that senesced  *
C      *                    during the day.  Units are g nutrients.    *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * LfSenesc  -Input - The amount of leaves that senesced during  *
C      *                    the day.  Units are g C.                   *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * LH        -Input - The array index for lignin to humus.       *
C      *                    Unitless.                                  *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * Lign      -Input - The array index for lignin.  Unitless.     *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * LignN     -Local - The amount of nitrogen in lignin in the    *
C      *                    leaves.  Units are g N.                    *
C      * Litter    -Output- A three dimensional matrix, of size 3 by 2 *
C      *                    by NLayers, which holds the amount of      *
C      *                    carbon and nitrogen as follows for each    *
C      *                    layer.  Units are g.                       *        
C      *                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
C      *                    ~                 CARBON    NITROGEN    ~  *
C      *                    ~                  (g C)      (g N)     ~  *
C      *                    ~    CELLULOSE      ##        ##        ~  *
C      *                    ~    EXTRACTIVES    ##        ##        ~  *
C      *                    ~    LIGNIN         ##        ##        ~  *
C      *                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
C      * Lyr       -Local - Index variable.  Unitless.                 *
C      * MaxSLTemp -Input - The maximal soil temperature for           *
C      *                    decomposition.  Units are deg C.           *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * NAmt      -Output- A matrix, of size 4 by NLayers, of the     *
C      *                    amount of nitrogen in  cellulose, lignin,  *
C      *                    extractives and humus for each layer.      *
C      *                    Units are g N/m**2.                        *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * NetNMin   -Local - Net nitrogen mineralization.  Units are    *
C      *                    g N/m**2/timestep.                         *
C      * Net_N_Min_Grams                                               *
C      *           -Local - The net nitrogen mineralization.  Units    *
C      *                    are g N/timestep.                          *
C      * N_Immob   -Local - A vector, if size 5, of nitrogen           *
C      *                    immobilization associated with each        *
C      *                    transformation.  Units are                 *
C      *                    g N/m**2/timestep.                         *
C      * N_Index   -Input - The index number for the component whose   *
C      *                    known total is to be updated by the        *
C      *                    mineralized nitrogen.  Unitless.           *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * N_In_OM   -Local - The amount of nitrogen in organic matter   *
C      *                    for the current layer.  Units are g N.     *
C      * N_Inorganic                                                   *
C      *           -Local - The amount of inorganic soil nitrogen.     *
C      *                    Units are g N/m**2.                        *
C      * NitInor   -Input - A vector of size NLayers, of the amount of *
C      *                    inorganic soil nitrogen  per layer.  Units *
C      *                    are g N.                                   *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * Nitr      -Input - An array index for nitrogen.  Unitless.    *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * NLayers   -Input - The number of soil layers in the system.   *
C      *                    Input from TREGRO, or read into the model  *
C      *                    by the routine InParam.  Unitless.         *
C      *                    (Common block VPos, file VModel.f)         *
C      * N_Mineral -Local - A vector, of size 5, of nitrogen           *
C      *                    mineralization associated with each        *
C      *                    transformation.  Units are                 *
C      *                    g N/m**2/timestep.                         *
C      * OptMoist  -Input - The optimal soil moisture.  Units are mm   *
C      *                    available water.                           *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * OptSLTemp -Input - The optimal soil temperature.  Units are   *
C      *                    deg C.                                     *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * QCurve    -Input - For decomposition, the q curvature         *
C      *                    parameter, calculated from the optimal and *
C      *                    maximal temperatures and the ratio between *
C      *                    the Q fcn when the temperature is optimal  *
C      *                    and when it is zero.  Unitless.            *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * Resp      -Local - A vector, of size 5, of respiration        *
C      *                    associated with each transformation.  Units*
C      *                    are g C/m**2/timestep.                     *
C      * RtNut     -Input - A matrix, NLayers by NComps, of the        *
C      *                    amount of nutrients in the roots, in each  *
C      *                    layer, that senesced during the day.  Input*
C      *                    from TREGRO, or read into the model by the *
C      *                    routine InParam.  Units are g nutr.        *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * RtSenesc  -Input - A vector, of size NLayers, of the amount of*
C      *                    roots, in each layer, that senesced during *
C      *                    the day.  Input from TREGRO, or read into  *
C      *                    the model by the routine InParam.  Units   *
C      *                    are g C.                                   *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * SkewMoist -Input - The soil moisture skewness parameter.      *
C      *                    Unitless.                                  *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * SLMoist   -Input - A vector, of size NLayers, of soil moisture*
C      *                    for each layer.  Input from TREGRO, or read*
C      *                    into the model by the routine InParam.     *
C      *                    Units are mm water.                        *
C      * SlTemp    -Input - A vector, of size NLayers, of soil         *
C      *                    temperatures for each layer.  Input from   *
C      *                    TREGRO, or read into the model by the      *
C      *                    routine InParam.  Units are deg C.         *                     
C      * TotResp   -Local - The sum of the respirations associated with*
C      *                    the transformations. Units are             *
C      *                    g C/m**2/timestep.                         *
C      * TranC     -Local - An array of the transformations of C from  *
C      *                    cellulose to extractives,                  *
C      *                    extractives to lignin,                     *
C      *                    humus to extractives,                      *
C      *                    lignin to extractives, and                 *
C      *                    lignin to humus.                           *
C      *                    Units are g C/m**2/timestep.               *
C      * TranEffic -Input - A vector, of size 5, of transformation     *
C      *                    efficiencies from                          *
C      *                    cellulose and extractives for all          *
C      *                    cellulose and extractives (g extr/g cell), *
C      *                    extractives and cellulose for all          *
C      *                    extractives and cellulose (g cell/g extr), *
C      *                    humus and extractives for all              *
C      *                    humus and extractives (g extr/g humu),     *
C      *                    lignin and extractives for all             *
C      *                    lignin and extractives (g extr/g lign),    *
C      *                    and lignin and humus for all               *
C      *                    and lignin and humus (g humu/g lign).      *
C      *                    (Common block VDecomp, file VModel.f)      *
C      *                                                               *
C      * TranRate  -Input - A vector, of size 5, of transformation     *
C      *                    rates from cellulose and extractives for   *
C      *                    all cellulose and extractives,             *
C      *                    extractives and cellulose, for all         *
C      *                    extractives and cellulose, humus and       *
C      *                    extractives, for all humus and extractives,*
C      *                    lignin and extractives, for all lignin and *
C      *                    extractives, and lignin and humus for all  *
C      *                    lignin and humus.  Units are 1/timestep.   *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * YASOut    -Input - The output file number for the YASEQL      *
C      *                    model.  Unitless.                          *
C      *                    (Common block VModel, file VModel.f)       *
C      *****************************************************************
       SUBROUTINE DECNTRL
       IMPLICIT NONE
       INCLUDE 'VMODEL.INC'
       INTEGER LYR
       REAL*8 CELLN, GNPERMOLE, LIGNN
       REAL*8 NETNMIN, NET_N_MIN_GRAMS, N_IMMOB(5), N_IN_OM
       REAL*8 N_INORGANIC, N_MINERAL(5)
       REAL*8 TRANC(5), RESP(5), TOTRESP
       PARAMETER (GNPERMOLE = 14.0) 


C      *--------------------------------------------------------*
C      * Perform the decomposition calculations for each layer. *
C      *--------------------------------------------------------*
       DO 100 LYR = 1, NLAYERS

C         *--------------------------------------------------------*
C         * Convert the inorganic nitrogen from moles to g N/m**2. *
C         *--------------------------------------------------------*
          N_INORGANIC = (NITINOR(LYR) * GNPERMOLE) / CATCHAREA(LYR)

C         *-------------------------------*
C         * Decompose what's in the soil. *
C         *-------------------------------*
          CALL DECOMP(
     >    CTON(CELL),    CTON(EXTR),    CTON(LIGN),    CTON(HUMU), 
     >    TRANEFFIC(CE), TRANEFFIC(EL), TRANEFFIC(HE), TRANEFFIC(LE), 
     >    TRANEFFIC(LH), CAMT(CELL,LYR),TRANRATE(CE),  TRANRATE(EL),
     >    TRANRATE(HE),  TRANRATE(LE),  TRANRATE(LH),  CAMT(EXTR,LYR),
     >    NAMT(EXTR,LYR),CAMT(HUMU,LYR),HALFSAT,       CAMT(LIGN,LYR),
     >    SLMOIST(LYR),  OPTMOIST,      BRDTHMOIST,    N_INORGANIC,
     >    QCURVE,        SKEWMOIST,     ILC(CELL),     ILC(EXTR),
     >    SLTEMP(LYR),   MAXSLTEMP,     OPTSLTEMP,     DCMPDETAIL,
     >    DCMPOUT,       LYR,           CATCHAREA(LYR),
     >    N_IMMOB(CE),   N_IMMOB(EL),   N_IMMOB(HE),   N_IMMOB(LE),
     >    N_IMMOB(LH),   N_MINERAL(CE), N_MINERAL(EL), N_MINERAL(HE),
     >    N_MINERAL(LE), N_MINERAL(LH), TRANC(CE),     TRANC(EL),
     >    TRANC(HE),     TRANC(LE),     TRANC(LH),     RESP(CE),
     >    RESP(EL),      RESP(HE),      RESP(LE),      RESP(LH),
     >    TOTRESP)

C         *-----------------------------------------------*
C         * Calculate the amount of carbon in cellulose,  *
C         * extractives and lignin in root litter.        *
C         *-----------------------------------------------*
          LITTER(CELL,CARB,LYR) = RTSENESC(LYR) * FRACTS(CELL,CARB,LYR)
          LITTER(EXTR,CARB,LYR) = RTSENESC(LYR) * FRACTS(EXTR,CARB,LYR)
          LITTER(LIGN,CARB,LYR) = RTSENESC(LYR) * FRACTS(LIGN,CARB,LYR)

C         *----------------------------------------------------------*
C         * Calculate the amount of nitrogen in cellulose and lignin *
C         * based on the amount of carbon and the carbon to nitrogen *
C         * ratio.                                                   *
C         *----------------------------------------------------------*
          LITTER(CELL,NITR,LYR) = LITTER(CELL,CARB,LYR) / CTON(CELL)
          LITTER(LIGN,NITR,LYR) = LITTER(LIGN,CARB,LYR) / CTON(LIGN)

C         *-----------------------------------------------------------*
C         * Calculate the amount of nitrogen in the extractives based *
C         * on the nitrogen pool and the amount in cellulose and      *
C         * lignin.                                                   *
C         *-----------------------------------------------------------*
          LITTER(EXTR,NITR,LYR) = 
     >       RTNUT(N_INDEX, LYR) - LITTER(CELL,NITR,LYR) - 
     >       LITTER(LIGN,NITR,LYR)
          IF (LITTER(EXTR,NITR,LYR).LT.0.0) THEN
             WRITE(6,*) 'Calculation of nitrogen for extractives has '
             WRITE(6,*) 'resulted in a negative number. '
             WRITE(6,*) 'Routine DeCntrl'
             WRITE(6,*) 'Layer = ', LYR
             WRITE(6,*) 'RtNitr = ',RTNUT(N_INDEX,LYR)
             WRITE(6,*) 'LITTER(CELL,NITR,LYR) = ',LITTER(CELL,NITR,LYR)
             WRITE(6,*) 'LITTER(LIGN,NITR,LYR) = ',LITTER(LIGN,NITR,LYR)
             WRITE(YASOUT,*) 
     >       'Calculation of nitrogen for extractives has '
             WRITE(YASOUT,*) 'resulted in a negative number. '
             WRITE(YASOUT,*) 'Routine DeCntrl'
             WRITE(YASOUT,*) 'Layer = ', LYR
             WRITE(YASOUT,*) 'RtNitr = ',RTNUT(N_INDEX,LYR)
             WRITE(YASOUT,*) 
     >       'LITTER(CELL,NITR,LYR) = ',LITTER(CELL,NITR,LYR)
             WRITE(YASOUT,*) 
     >       'LITTER(LIGN,NITR,LYR) = ',LITTER(LIGN,NITR,LYR)
             STOP
          ENDIF

C         *-----------------------------------------------------------*
C         * If this is the top layer, add the carbon and nitrogen     *
C         * contributions from the dead leaves to the total senesced. *
C         * Nitrogen is calculated using the carbon amount and the    *
C         * carbon-to-nitrogen ratio for cellulose and lignin and by  *
C         * difference for extractives.                               *
C         *-----------------------------------------------------------*
          IF (LYR.EQ.1) THEN
             LITTER(CELL,CARB,LYR) = LITTER(CELL,CARB,LYR) + 
     >                               LFSENESC * LFFRACTS(CELL,CARB)
             LITTER(EXTR,CARB,LYR) = LITTER(EXTR,CARB,LYR) +
     >                               LFSENESC * LFFRACTS(EXTR,CARB)
             LITTER(LIGN,CARB,LYR) = LITTER(LIGN,CARB,LYR) + 
     >                               LFSENESC * LFFRACTS(LIGN,CARB)
 
             CELLN = LFSENESC * LFFRACTS(CELL,CARB) / CTON(CELL)
             LIGNN = LFSENESC * LFFRACTS(LIGN,CARB) / CTON(LIGN)

             LITTER(CELL,NITR,LYR) = LITTER(CELL,NITR,LYR) + CELLN
             LITTER(LIGN,NITR,LYR) = LITTER(LIGN,NITR,LYR) + LIGNN

             LITTER(EXTR,NITR,LYR) = LITTER(EXTR,NITR,LYR) +
     >       (LFNUT(N_INDEX) - CELLN - LIGNN) 
             IF (LITTER(EXTR,NITR,LYR).LT.0.0) THEN
                WRITE(6,*) 'Calculation of nitrogen ',
     >          'for extractives has resulted in a negative number. '
                WRITE(6,*) 'Routine DeCntrl'
                WRITE(6,*) 'Layer = ', LYR
                WRITE(6,*) 'LfNitr = ',LFNUT(N_INDEX)
                WRITE(6,*) 'CELLN = ',CELLN
                WRITE(6,*) 'LIGNN = ',LIGNN
                WRITE(YASOUT,*) 'Calculation of nitrogen ',
     >          'for extractives has resulted in a negative number. '
                WRITE(YASOUT,*) 'Routine DeCntrl'
                WRITE(YASOUT,*) 'Layer = ', LYR
                WRITE(YASOUT,*) 'LfNitr = ',LFNUT(N_INDEX)
                WRITE(YASOUT,*) 'CELLN = ',CELLN
                WRITE(YASOUT,*) 'LIGNN = ',LIGNN
                STOP
             ENDIF

          ENDIF

C         *---------------------------------------------------*
C         * Update carbon amounts for cellulose, extractives, *
C         * lignin and humus.                                 *
C         *---------------------------------------------------*
          CAMT(CELL,LYR) = CAMT(CELL,LYR) + LITTER(CELL, CARB, LYR) /
     >                     CATCHAREA(LYR) - TRANC(CE) - RESP(CE)
          CAMT(EXTR,LYR) = CAMT(EXTR,LYR) + LITTER(EXTR, CARB, LYR) / 
     >                     CATCHAREA(LYR) + TRANC(CE) + TRANC(LE) + 
     >                     TRANC(HE) - TRANC(EL) - RESP(EL)
          CAMT(LIGN,LYR) = CAMT(LIGN,LYR) + LITTER(LIGN, CARB, LYR) / 
     >                     CATCHAREA(LYR) + TRANC(EL) - TRANC(LE) - 
     >                     TRANC(LH) - RESP(LE) - RESP(LH)
          CAMT(HUMU,LYR) = CAMT(HUMU,LYR) + TRANC(LH) - TRANC(HE) 
     >                     - RESP(HE)

C         *-----------------------------------------------------*
C         * Update nitrogen amounts for cellulose, extractives, *
C         * lignin and humus.                                   *
C         *-----------------------------------------------------*
          NAMT(CELL,LYR) = NAMT(CELL,LYR) + LITTER(CELL, NITR, LYR) /
     >                     CATCHAREA(LYR) - N_MINERAL(CE)
          NAMT(EXTR,LYR) = NAMT(EXTR,LYR) + LITTER(EXTR, NITR, LYR) /
     >                     CATCHAREA(LYR) + N_IMMOB(CE) + 
     >                     N_IMMOB(LE) + N_IMMOB(HE) - N_MINERAL(EL)
          NAMT(LIGN,LYR) = NAMT(LIGN,LYR) + LITTER(LIGN, NITR, LYR) /
     >                     CATCHAREA(LYR) + N_IMMOB(EL) - 
     >                     N_MINERAL(LE) - N_MINERAL(LH)
          NAMT(HUMU,LYR) = NAMT(HUMU,LYR) + N_IMMOB(LH) - N_MINERAL(HE)

C         *---------------------------------*
C         * Calculate net N mineralization. *
C         *---------------------------------*
          NETNMIN = N_MINERAL(CE) +  N_MINERAL(EL) +  N_MINERAL(HE) + 
     >              N_MINERAL(LE) +  N_MINERAL(LH) - 
     >              N_IMMOB(CE)   -  N_IMMOB(EL)   -  N_IMMOB(HE) - 
     >              N_IMMOB(LE)   -  N_IMMOB(LH) 

C         *-------------------------------------------------------------*
C         * Update soil inorganic nitrogen.  Need to convert net        *
C         * mineralization from g N/m**2 to moles.                      *
C         * Don't allow the KT value to go negative.  If it does, stop. *
C         *-------------------------------------------------------------*
          KT(N_INDEX,LYR) = KT(N_INDEX,LYR) + 
     >                     (NETNMIN / GNPERMOLE) * CATCHAREA(LYR)
          IF (KT(N_INDEX,LYR).LT.0.0) THEN
             WRITE(6,*) 
     >       'The known total for nitrogen as calculated in'
             WRITE(6,*) 'DeCntrl is less than zero'
             WRITE(6,*) 'NetNMin = ',NETNMIN
             WRITE(6,*) 'Layer   = ',LYR
             WRITE(YASOUT,*) 
     >       'The known total for nitrogen as calculated in'
             WRITE(YASOUT,*) 'DeCntrl is less than zero'
             WRITE(YASOUT,*) 'NetNMin = ',NETNMIN
             WRITE(YASOUT,*) 'Layer   = ',LYR
             STOP
          ENDIF
          NITINOR(LYR) = KT(N_INDEX, LYR)

C         *---------------------------------------------------*
C         * Do a first order decay for calcium, magnesium and *
C         * potassium.                                        *
C         *---------------------------------------------------*
          CALL FODECAY(LYR)

C         *-------------------------------------------------*
C         * Do a decay that is proportional to nitrogen for *
C         * phosphorus and sulfur.                          *
C         *-------------------------------------------------*
          N_IN_OM = NAMT(CELL,LYR) + NAMT(EXTR,LYR) + NAMT(LIGN,LYR) +
     >              NAMT(HUMU,LYR) 
          N_IN_OM = N_IN_OM * CATCHAREA(LYR)
          NET_N_MIN_GRAMS = NETNMIN * CATCHAREA(LYR)
          CALL PROPDECAY(LYR, N_IN_OM, NET_N_MIN_GRAMS)
  100  CONTINUE

	      RETURN
	      END
C      *****************************************************************
C      *                    END SUBROUTINE                             *
C      *****************************************************************
